<?php
// Start session only if not already active
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Initialize variables
 $ptitle = "Investment Packages";
 $username = $_SESSION['username'] ?? '';
 $userId = null; // Will be set dynamically

// Include required files
require $_SERVER['DOCUMENT_ROOT']."/stream.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/includes/generalinclude.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/user/includes/generalinclude.php";

// Set timezone
date_default_timezone_set('UTC');

// Fix database structure for transactions table
 $fixTransactionTable = $link->query("ALTER TABLE flow_transactions MODIFY COLUMN type VARCHAR(20)");
if ($fixTransactionTable === false) {
    // Log error but don't stop execution
    error_log("Error altering flow_transactions table: " . $link->error);
}

// Get user ID from session or database
// First, try to get user ID from session
if (isset($_SESSION['user_id'])) {
    $userId = $_SESSION['user_id'];
} 
// If not in session, try to get from username in session
elseif (isset($_SESSION['username'])) {
    $username = $_SESSION['username'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE username = ?");
    $getUserQuery->bind_param("s", $username);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}
// If still not found, try to get from email in session
elseif (isset($_SESSION['email'])) {
    $email = $_SESSION['email'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE email = ?");
    $getUserQuery->bind_param("s", $email);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}

// If user ID still not found, create a default user for testing
if (!$userId) {
    // Check if default user exists
    $checkUserQuery = $link->prepare("SELECT id FROM users WHERE username = 'Test User'");
    $checkUserQuery->execute();
    $checkResult = $checkUserQuery->get_result();
    
    if ($checkResult->num_rows == 0) {
        // Create default user
        $insertUserQuery = $link->prepare("INSERT INTO users (username, email, password, funds) VALUES (?, ?, ?, ?)");
        $defaultUsername = 'Test User';
        $defaultEmail = 'test@example.com';
        $defaultPassword = password_hash('password123', PASSWORD_DEFAULT);
        $defaultFunds = 0.00;
        $insertUserQuery->bind_param("sssd", $defaultUsername, $defaultEmail, $defaultPassword, $defaultFunds);
        $insertUserQuery->execute();
        $userId = $link->insert_id;
    } else {
        $userData = $checkResult->fetch_assoc();
        $userId = $userData['id'];
    }
    
    // Set user_id in session
    $_SESSION['user_id'] = $userId;
    $_SESSION['username'] = 'Test User';
    $_SESSION['email'] = 'test@example.com';
}

// Check if flow_user_packages table exists, create if not
 $tableCheck = $link->query("SHOW TABLES LIKE 'flow_user_packages'");
if ($tableCheck->num_rows == 0) {
    $createTable = "CREATE TABLE `flow_user_packages` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `user_id` int(11) NOT NULL,
        `package_id` int(11) NOT NULL,
        `start_date` date NOT NULL,
        `end_date` date NOT NULL,
        `total_earned` decimal(10,2) DEFAULT 0,
        `status` enum('active','completed') DEFAULT 'active',
        `last_claim_date` datetime DEFAULT NULL,
        PRIMARY KEY (`id`),
        FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
        FOREIGN KEY (`package_id`) REFERENCES `flow_packages`(`id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
    
    if ($link->query($createTable)) {
        $pkgMsg = '<div class="alert alert-info">User packages table created successfully!</div>';
    } else {
        die("Error creating user packages table: " . $link->error);
    }
} else {
    // Check if last_claim_date column exists and is datetime
    $columnCheck = $link->query("SHOW COLUMNS FROM flow_user_packages LIKE 'last_claim_date'");
    if ($columnCheck->num_rows > 0) {
        $columnInfo = $columnCheck->fetch_assoc();
        if ($columnInfo['Type'] !== 'datetime') {
            // Alter column to datetime
            $alterTable = "ALTER TABLE `flow_user_packages` MODIFY COLUMN `last_claim_date` datetime DEFAULT NULL";
            $link->query($alterTable);
        }
    } else {
        // Add the column as datetime
        $alterTable = "ALTER TABLE `flow_user_packages` ADD COLUMN `last_claim_date` datetime DEFAULT NULL";
        $link->query($alterTable);
    }
}

// Update expired packages
 $updateExpired = $link->prepare("UPDATE flow_user_packages SET status = 'completed' WHERE end_date < CURDATE() AND status = 'active'");
 $updateExpired->execute();

// Handle package subscription
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['subscribePackage'])) {
    $pkgId = $_POST['pkgId'];
    
    // Get package details
    $pkgQuery = $link->prepare("SELECT * FROM flow_packages WHERE id = ?");
    $pkgQuery->bind_param("i", $pkgId);
    $pkgQuery->execute();
    $pkgResult = $pkgQuery->get_result();
    
    if ($pkgResult->num_rows > 0) {
        $package = $pkgResult->fetch_assoc();
        $amount = $package['amount'];
        $duration = $package['duration'];
        
        // Check user balance
        $balanceQuery = $link->prepare("SELECT funds FROM users WHERE id = ?");
        $balanceQuery->bind_param("i", $userId);
        $balanceQuery->execute();
        $balanceResult = $balanceQuery->get_result();
        $user = $balanceResult->fetch_assoc();
        $balance = $user['funds'];
        
        if ($balance >= $amount) {
            // Deduct amount from user balance
            $updateBalance = $link->prepare("UPDATE users SET funds = funds - ? WHERE id = ?");
            $updateBalance->bind_param("di", $amount, $userId);
            $updateBalance->execute();
            
            // Calculate dates
            $startDate = date('Y-m-d');
            $endDate = date('Y-m-d', strtotime("+$duration days"));
            
            // Insert subscription
            $insertSub = $link->prepare("INSERT INTO flow_user_packages (user_id, package_id, start_date, end_date) VALUES (?, ?, ?, ?)");
            $insertSub->bind_param("iiss", $userId, $pkgId, $startDate, $endDate);
            $insertSub->execute();
            
            // Record transaction
            $transactionId = 'PKG' . time() . rand(1000, 9999);
            $description = "Package subscription: " . $package['name'];
            $insertTransaction = $link->prepare("INSERT INTO flow_transactions (user_id, transaction_id, amount, type, description, created_at) VALUES (?, ?, ?, 'package', ?, NOW())");
            $insertTransaction->bind_param("idss", $userId, $transactionId, $amount, $description);
            $insertTransaction->execute();
            
            $pkgMsg = '<div class="alert alert-success">Package subscribed successfully! ₦' . number_format($amount, 2) . ' has been deducted from your balance.</div>';
        } else {
            $pkgMsg = '<div class="alert alert-danger">Insufficient balance! You need ₦' . number_format($amount, 2) . ' but have only ₦' . number_format($balance, 2) . '</div>';
        }
    } else {
        $pkgMsg = '<div class="alert alert-danger">Invalid package selected!</div>';
    }
}

// Handle daily income claim
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['claimDailyIncome'])) {
    $subscriptionId = $_POST['subscriptionId'];
    
    // Get the subscription details
    $subQuery = $link->prepare("SELECT up.*, p.daily_income FROM flow_user_packages up JOIN flow_packages p ON up.package_id = p.id WHERE up.id = ? AND up.user_id = ? AND up.status = 'active'");
    $subQuery->bind_param("ii", $subscriptionId, $userId);
    $subQuery->execute();
    $subResult = $subQuery->get_result();
    
    if ($subResult->num_rows > 0) {
        $subscription = $subResult->fetch_assoc();
        $dailyIncome = $subscription['daily_income'];
        $lastClaimDate = $subscription['last_claim_date'];
        $now = new DateTime();
        
        // Check if it's the first claim or if at least 24 hours have passed since the last claim
        if ($lastClaimDate === null) {
            // First claim
            $canClaim = true;
        } else {
            $lastClaim = new DateTime($lastClaimDate);
            $lastClaim->modify('+24 hours');
            $canClaim = ($now >= $lastClaim);
        }
        
        if ($canClaim) {
            // Add daily income to user's funds
            $updateFunds = $link->prepare("UPDATE users SET funds = funds + ? WHERE id = ?");
            $updateFunds->bind_param("di", $dailyIncome, $userId);
            $updateFunds->execute();
            
            // Update the subscription: set last_claim_date to now and add to total_earned
            $updateSubscription = $link->prepare("UPDATE flow_user_packages SET last_claim_date = NOW(), total_earned = total_earned + ? WHERE id = ?");
            $updateSubscription->bind_param("di", $dailyIncome, $subscriptionId);
            $updateSubscription->execute();
            
            // Record transaction
            $transactionId = 'INC' . time() . rand(1000, 9999);
            $description = "Daily income from package";
            $insertTransaction = $link->prepare("INSERT INTO flow_transactions (user_id, transaction_id, amount, type, description, created_at) VALUES (?, ?, ?, 'income', ?, NOW())");
            $insertTransaction->bind_param("idss", $userId, $transactionId, $dailyIncome, $description);
            $insertTransaction->execute();
            
            $pkgMsg = '<div class="alert alert-success">Daily income claimed successfully! ₦' . number_format($dailyIncome, 2) . ' has been added to your balance.</div>';
        } else {
            // Calculate time until next claim
            $lastClaimTime = new DateTime($lastClaimDate);
            $lastClaimTime->modify('+24 hours');
            $timeUntilNextClaim = $lastClaimTime->getTimestamp() - $now->getTimestamp();
            $hoursUntil = floor($timeUntilNextClaim / 3600);
            $minutesUntil = floor(($timeUntilNextClaim % 3600) / 60);
            $secondsUntil = floor(($timeUntilNextClaim % 60));
            
            $pkgMsg = '<div class="alert alert-danger">You can only claim once per day! Next claim available in ' . 
                     $hoursUntil . ' hours, ' . $minutesUntil . ' minutes, and ' . $secondsUntil . ' seconds.</div>';
        }
    } else {
        $pkgMsg = '<div class="alert alert-danger">Invalid subscription or package is not active!</div>';
    }
}

// Get user's active packages
 $activeQuery = $link->prepare("
    SELECT up.*, p.name, p.amount, p.daily_income, p.total_income, p.min_withdrawal 
    FROM flow_user_packages up 
    JOIN flow_packages p ON up.package_id = p.id 
    WHERE up.user_id = ? AND up.status = 'active'
    ORDER BY up.id DESC
");
 $activeQuery->bind_param("i", $userId);
 $activeQuery->execute();
 $activeResult = $activeQuery->get_result();

// Get user's expired packages
 $expiredQuery = $link->prepare("
    SELECT up.*, p.name, p.amount, p.daily_income, p.total_income 
    FROM flow_user_packages up 
    JOIN flow_packages p ON up.package_id = p.id 
    WHERE up.user_id = ? AND up.status = 'completed'
    ORDER BY up.id DESC
");
 $expiredQuery->bind_param("i", $userId);
 $expiredQuery->execute();
 $expiredResult = $expiredQuery->get_result();

// Get packages that the user has never subscribed to
 $availableQuery = $link->prepare("
    SELECT p.* 
    FROM flow_packages p
    WHERE p.id NOT IN (
        SELECT package_id FROM flow_user_packages WHERE user_id = ?
    )
    ORDER BY p.id DESC
");
 $availableQuery->bind_param("i", $userId);
 $availableQuery->execute();
 $availableResult = $availableQuery->get_result();

// Get all packages
 $allPackagesQuery = $link->prepare("SELECT * FROM flow_packages ORDER BY id DESC");
 $allPackagesQuery->execute();
 $allPackagesResult = $allPackagesQuery->get_result();

// Include header
include "includes/header.php";
?>

<style>
  :root {
    --primary-color: #4e73df;
    --primary-dark: #2e59d9;
    --primary-light: #f8f9fc;
    --text-color: #333;
    --bg-color: #f8f9fc;
    --card-bg: #ffffff;
    --card-border: #e3e6f0;
    --shadow: rgba(0, 0, 0, 0.1);
    --card-radius: 12px;
    --transition: all 0.3s ease;
  }

  body {
    position: relative;
    margin: 0;
    padding: 0;
    overflow-x: hidden;
    color: var(--text-color);
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    background-color: var(--bg-color);
  }

  .nk-content {
    padding: 2rem 0;
  }

  .nk-block-head {
    margin-bottom: 2.5rem;
  }

  .nk-block-title {
    font-size: 2.5rem;
    font-weight: 700;
    color: var(--text-color);
    margin-bottom: 0.5rem;
  }

  .packages-container {
    width: 100%;
    margin: 30px 0;
    padding: 0 15px;
  }

  .packages-container h3 {
    font-size: 1.8rem;
    font-weight: 600;
    color: var(--text-color);
    margin-bottom: 1.5rem;
    padding-bottom: 0.75rem;
    border-bottom: 2px solid var(--primary-color);
    display: inline-block;
  }

  .packages-row {
    display: flex;
    flex-wrap: wrap;
    gap: 25px;
    margin-bottom: 40px;
  }

  .package-card {
    flex: 0 0 calc(33.333% - 25px);
    background-color: var(--card-bg);
    color: var(--text-color);
    border-radius: var(--card-radius);
    box-shadow: 0 5px 15px var(--shadow);
    padding: 25px;
    border: 2px solid var(--card-border);
    transition: var(--transition);
    position: relative;
    overflow: hidden;
    display: flex;
    flex-direction: column;
  }

  .package-card::before {
    content: "";
    position: absolute;
    top: 0;
    left: 0;
    width: 5px;
    height: 100%;
    background-color: var(--primary-color);
    z-index: 1;
  }

  .package-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 25px var(--shadow);
    border-color: var(--primary-dark);
  }

  .package-header {
    margin-bottom: 20px;
    position: relative;
    z-index: 2;
  }

  .package-title {
    font-weight: 700;
    font-size: 1.5rem;
    color: var(--text-color);
    margin: 0 0 10px 0;
    line-height: 1.3;
  }

  .package-amount {
    font-size: 1.8rem;
    font-weight: 700;
    color: var(--primary-color);
    margin: 15px 0;
    display: flex;
    align-items: center;
  }

  .package-amount::before {
    content: "₦";
    margin-right: 5px;
    font-size: 1.2rem;
  }

  .package-detail {
    display: flex;
    justify-content: space-between;
    margin-bottom: 10px;
    font-size: 0.95rem;
  }

  .package-detail-label {
    font-weight: 600;
    color: #6c757d;
  }

  .package-detail-value {
    font-weight: 500;
  }

  .package-footer {
    margin-top: auto;
    position: relative;
    z-index: 2;
  }

  .package-btn {
    width: 100%;
    padding: 12px 16px;
    background-color: var(--primary-color);
    color: white;
    border: none;
    border-radius: 30px;
    cursor: pointer;
    transition: var(--transition);
    font-weight: 600;
    font-size: 1rem;
    box-shadow: 0 4px 8px var(--shadow);
  }

  .package-btn:hover {
    background-color: var(--primary-dark);
    transform: translateY(-3px);
    box-shadow: 0 6px 12px var(--shadow);
  }

  .package-btn:disabled {
    background-color: #6c757d;
    cursor: not-allowed;
    transform: none;
  }

  .no-package {
    text-align: center;
    font-size: 1.2rem;
    color: var(--text-color);
    padding: 30px;
    background-color: var(--primary-light);
    border-radius: var(--card-radius);
    margin: 20px 0;
    border: 1px dashed var(--primary-color);
  }

  .section-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
  }

  .toggle-btn {
    background-color: var(--primary-color);
    border: none;
    color: white;
    padding: 8px 16px;
    border-radius: 30px;
    cursor: pointer;
    transition: var(--transition);
    font-weight: 500;
    box-shadow: 0 4px 8px var(--shadow);
  }

  .toggle-btn:hover {
    background-color: var(--primary-dark);
  }

  .user-package-card {
    background-color: var(--primary-light);
    border: 2px solid var(--primary-color);
  }

  .progress-container {
    margin: 15px 0;
  }

  .progress {
    height: 10px;
    border-radius: 5px;
    background-color: #e9ecef;
  }

  .progress-bar {
    height: 100%;
    border-radius: 5px;
    background-color: var(--primary-color);
  }

  .days-left {
    font-size: 0.9rem;
    color: #6c757d;
    margin-top: 5px;
    text-align: right;
  }

  .days-left div {
    margin-bottom: 5px;
  }

  .claim-status {
    font-weight: 600;
    color: var(--primary-color);
  }

  .countdown {
    font-size: 0.85rem;
    color: var(--primary-color);
    font-weight: 600;
  }

  .refresh-btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 30px;
    height: 30px;
    border-radius: 50%;
    background-color: var(--primary-color);
    color: white;
    border: none;
    cursor: pointer;
    margin-left: 10px;
    transition: var(--transition);
  }

  .refresh-btn:hover {
    background-color: var(--primary-dark);
    transform: rotate(180deg);
  }

  .completed-badge {
    position: absolute;
    top: 15px;
    right: 15px;
    background-color: #28a745;
    color: white;
    padding: 5px 10px;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
  }
  
  .nav-tabs {
    margin-bottom: 20px;
    border-bottom: 2px solid var(--primary-color);
  }
  
  .nav-tabs .nav-link {
    color: #6c757d;
    font-weight: 500;
    border: none;
    border-bottom: 2px solid transparent;
    border-radius: 0;
    padding: 10px 20px;
    transition: var(--transition);
  }
  
  .nav-tabs .nav-link:hover {
    border-color: transparent;
    color: var(--primary-color);
  }
  
  .nav-tabs .nav-link.active {
    color: var(--primary-color);
    background-color: transparent;
    border-color: var(--primary-color);
  }
  
  .tab-content {
    padding: 20px 0;
  }
  
  .claim-info {
    font-size: 0.85rem;
    color: #6c757d;
    margin-top: 5px;
    text-align: center;
  }
  
  .countdown-container {
    display: flex;
    align-items: center;
    justify-content: flex-end;
  }
  
  @media (max-width: 1200px) {
    .package-card {
      flex: 0 0 calc(50% - 25px);
    }
  }
  
  @media (max-width: 768px) {
    .package-card {
      flex: 0 0 100%;
    }
    
    .nk-block-title {
      font-size: 2rem;
    }
    
    .packages-container {
      padding: 0 10px;
    }
  }
</style>

<div class="nk-content">
    <div class="container-fluid">
        <div class="nk-content-inner">
            <div class="nk-content-body">
                <div class="nk-block-head">
                    <div class="nk-block-head-between flex-wrap gap g-2">
                        <br>
                    </div>
                </div>
                
                <?php if (isset($pkgMsg)) echo $pkgMsg; ?>
                
                <div class="packages-container">
                    <!-- Tab Navigation -->
                    <ul class="nav nav-tabs" id="packageTabs" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" id="active-tab" data-bs-toggle="tab" data-bs-target="#active" type="button" role="tab" aria-controls="active" aria-selected="true">Active Packages</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="expired-tab" data-bs-toggle="tab" data-bs-target="#expired" type="button" role="tab" aria-controls="expired" aria-selected="false">Expired Packages</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="available-tab" data-bs-toggle="tab" data-bs-target="#available" type="button" role="tab" aria-controls="available" aria-selected="false">Available Packages</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="all-tab" data-bs-toggle="tab" data-bs-target="#all" type="button" role="tab" aria-controls="all" aria-selected="false">All Packages</button>
                        </li>
                    </ul>

                    <!-- Tab Content -->
                    <div class="tab-content" id="packageTabsContent">
                        <!-- Active Packages Tab -->
                        <div class="tab-pane fade show active" id="active" role="tabpanel" aria-labelledby="active-tab">
                            <div class="packages-row">
                                <?php if ($activeResult->num_rows > 0): ?>
                                    <?php while ($pkg = $activeResult->fetch_assoc()): ?>
                                        <?php
                                            $id = $pkg['id'];
                                            $name = $pkg['name'];
                                            $amount = $pkg['amount'];
                                            $daily_income = $pkg['daily_income'];
                                            $total_income = $pkg['total_income'];
                                            $start_date = $pkg['start_date'];
                                            $end_date = $pkg['end_date'];
                                            $total_earned = $pkg['total_earned'];
                                            $last_claim_date = $pkg['last_claim_date'];
                                            
                                            // Calculate days left and progress
                                            $today = new DateTime();
                                            $end = new DateTime($end_date);
                                            $start = new DateTime($start_date);
                                            $days_total = $start->diff($end)->days;
                                            $days_left = $today->diff($end)->days;
                                            $days_left = $days_left > 0 ? $days_left : 0;
                                            $progress = (($days_total - $days_left) / $days_total) * 100;
                                            
                                            // Calculate expected earnings
                                            $expected_earnings = $daily_income * ($days_total - $days_left);
                                            
                                            // Check if user can claim today
                                            $canClaim = false;
                                            if ($last_claim_date === null) {
                                                // First claim
                                                $canClaim = true;
                                            } else {
                                                // Check if at least 24 hours have passed
                                                $lastClaim = new DateTime($last_claim_date);
                                                $lastClaim->modify('+24 hours');
                                                $canClaim = ($today >= $lastClaim);
                                            }
                                            
                                            // Calculate next claim time for countdown
                                            $nextClaimTime = null;
                                            if (!$canClaim) {
                                                $lastClaimDateObj = new DateTime($last_claim_date);
                                                $lastClaimDateObj->modify('+24 hours');
                                                $nextClaimTime = $lastClaimDateObj->getTimestamp();
                                            }
                                        ?>
                                        <div class="package-card user-package-card">
                                            <div class="package-header">
                                                <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                            </div>
                                            <div class="package-body">
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Daily Income:</span>
                                                    <span class="package-detail-value">₦<?= number_format($daily_income, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Total Return:</span>
                                                    <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Earned So Far:</span>
                                                    <span class="package-detail-value">₦<?= number_format($total_earned, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Expected Earnings:</span>
                                                    <span class="package-detail-value">₦<?= number_format($expected_earnings, 2) ?></span>
                                                </div>
                                                
                                                <div class="progress-container">
                                                    <div class="progress">
                                                        <div class="progress-bar" style="width: <?= $progress ?>%"></div>
                                                    </div>
                                                    <div class="days-left">
                                                        <div><?= $days_left ?> days left</div>
                                                        <div class="claim-status">Claim status: <?= $canClaim ? '0/1 (Unclaimed)' : '1/1 (Claimed)' ?></div>
                                                        <?php if (!$canClaim && $nextClaimTime): ?>
                                                            <div class="countdown-container">
                                                                <div class="countdown" id="countdown-<?= $id ?>" data-target="<?= $nextClaimTime ?>">
                                                                    Next claim in: <span>23:59:59</span>
                                                                </div>
                                                                <button class="refresh-btn" onclick="location.reload()" title="Refresh page">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                                                                        <path d="M8 3a5 5 0 1 0 4.546 2.914.5.5 0 0 1 .908-.417A6 6 0 1 1 8 2v1z"/>
                                                                        <path d="M8 4.466V.534a.25.25 0 0 1 .41-.192l2.36 1.966c.12.1.12.284 0 .384L8.41 4.658A.25.25 0 0 1 8 4.466z"/>
                                                                    </svg>
                                                                </button>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="package-footer">
                                                <?php if ($canClaim): ?>
                                                    <form method="POST">
                                                        <input type="hidden" name="subscriptionId" value="<?= $id ?>">
                                                        <button type="submit" name="claimDailyIncome" class="package-btn">Claim Daily Income (₦<?= number_format($daily_income, 2) ?>)</button>
                                                    </form>
                                                <?php else: ?>
                                                    <button class="package-btn" disabled>Claim Daily Income (₦<?= number_format($daily_income, 2) ?>)</button>
                                                    <div class="claim-info">Next claim available tomorrow</div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <div class="no-package">You don't have any active packages.</div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Expired Packages Tab -->
                        <div class="tab-pane fade" id="expired" role="tabpanel" aria-labelledby="expired-tab">
                            <div class="packages-row">
                                <?php if ($expiredResult->num_rows > 0): ?>
                                    <?php while ($pkg = $expiredResult->fetch_assoc()): ?>
                                        <?php
                                            $id = $pkg['id'];
                                            $name = $pkg['name'];
                                            $amount = $pkg['amount'];
                                            $total_income = $pkg['total_income'];
                                            $total_earned = $pkg['total_earned'];
                                            $start_date = $pkg['start_date'];
                                            $end_date = $pkg['end_date'];
                                            $package_id = $pkg['package_id'];
                                            
                                            // Calculate profit
                                            $profit = $total_income - $amount;
                                        ?>
                                        <div class="package-card user-package-card">
                                            <div class="completed-badge">COMPLETED</div>
                                            <div class="package-header">
                                                <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                            </div>
                                            <div class="package-body">
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Total Return:</span>
                                                    <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Total Earned:</span>
                                                    <span class="package-detail-value">₦<?= number_format($total_earned, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Net Profit:</span>
                                                    <span class="package-detail-value">₦<?= number_format($profit, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Duration:</span>
                                                    <span class="package-detail-value"><?= date('M j, Y', strtotime($start_date)) ?> - <?= date('M j, Y', strtotime($end_date)) ?></span>
                                                </div>
                                            </div>
                                            <div class="package-footer">
                                                <form method="POST">
                                                    <input type="hidden" name="pkgId" value="<?= $package_id ?>">
                                                    <button type="submit" name="subscribePackage" class="package-btn">Resubscribe Package</button>
                                                </form>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <div class="no-package">You haven't completed any packages yet.</div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Available Packages Tab -->
                        <div class="tab-pane fade" id="available" role="tabpanel" aria-labelledby="available-tab">
                            <div class="packages-row">
                                <?php if ($availableResult->num_rows > 0): ?>
                                    <?php while ($package = $availableResult->fetch_assoc()): ?>
                                        <?php
                                            $id = $package['id'];
                                            $name = $package['name'];
                                            $amount = $package['amount'];
                                            $daily_income = $package['daily_income'];
                                            $duration = $package['duration'];
                                            $total_income = $package['total_income'];
                                            $min_withdrawal = $package['min_withdrawal'];
                                            $referral_bonus = $package['referral_bonus'];
                                            
                                            // Calculate ROI
                                            $roi = (($total_income - $amount) / $amount) * 100;
                                        ?>
                                        <div class="package-card">
                                            <div class="package-header">
                                                <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                            </div>
                                            <div class="package-body">
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Daily Income:</span>
                                                    <span class="package-detail-value">₦<?= number_format($daily_income, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Duration:</span>
                                                    <span class="package-detail-value"><?= $duration ?> days</span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Total Return:</span>
                                                    <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">ROI:</span>
                                                    <span class="package-detail-value"><?= number_format($roi, 1) ?>%</span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Min. Withdrawal:</span>
                                                    <span class="package-detail-value">₦<?= number_format($min_withdrawal, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Referral Bonus:</span>
                                                    <span class="package-detail-value">₦<?= number_format($referral_bonus, 2) ?></span>
                                                </div>
                                            </div>
                                            <div class="package-footer">
                                                <form method="POST">
                                                    <input type="hidden" name="pkgId" value="<?= $id ?>">
                                                    <button type="submit" name="subscribePackage" class="package-btn">Subscribe Now</button>
                                                </form>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <div class="no-package">No available packages at the moment.</div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- All Packages Tab -->
                        <div class="tab-pane fade" id="all" role="tabpanel" aria-labelledby="all-tab">
                            <div class="packages-row">
                                <?php if ($allPackagesResult->num_rows > 0): ?>
                                    <?php while ($package = $allPackagesResult->fetch_assoc()): ?>
                                        <?php
                                            $id = $package['id'];
                                            $name = $package['name'];
                                            $amount = $package['amount'];
                                            $daily_income = $package['daily_income'];
                                            $duration = $package['duration'];
                                            $total_income = $package['total_income'];
                                            $min_withdrawal = $package['min_withdrawal'];
                                            $referral_bonus = $package['referral_bonus'];
                                            
                                            // Calculate ROI
                                            $roi = (($total_income - $amount) / $amount) * 100;
                                            
                                            // Check if user has active subscription for this package
                                            $hasActive = false;
                                            $checkActive = $link->prepare("SELECT id FROM flow_user_packages WHERE user_id = ? AND package_id = ? AND status = 'active'");
                                            $checkActive->bind_param("ii", $userId, $id);
                                            $checkActive->execute();
                                            $activeCheck = $checkActive->get_result();
                                            if ($activeCheck->num_rows > 0) {
                                                $hasActive = true;
                                            }
                                        ?>
                                        <div class="package-card">
                                            <div class="package-header">
                                                <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                            </div>
                                            <div class="package-body">
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Daily Income:</span>
                                                    <span class="package-detail-value">₦<?= number_format($daily_income, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Duration:</span>
                                                    <span class="package-detail-value"><?= $duration ?> days</span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Total Return:</span>
                                                    <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">ROI:</span>
                                                    <span class="package-detail-value"><?= number_format($roi, 1) ?>%</span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Min. Withdrawal:</span>
                                                    <span class="package-detail-value">₦<?= number_format($min_withdrawal, 2) ?></span>
                                                </div>
                                                <div class="package-detail">
                                                    <span class="package-detail-label">Referral Bonus:</span>
                                                    <span class="package-detail-value">₦<?= number_format($referral_bonus, 2) ?></span>
                                                </div>
                                            </div>
                                            <div class="package-footer">
                                                <?php if ($hasActive): ?>
                                                    <button class="package-btn" disabled>Already Subscribed</button>
                                                <?php else: ?>
                                                    <form method="POST">
                                                        <input type="hidden" name="pkgId" value="<?= $id ?>">
                                                        <button type="submit" name="subscribePackage" class="package-btn">Subscribe Now</button>
                                                    </form>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <div class="no-package">No packages available.</div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Function to update all countdowns
    function updateCountdowns() {
        // Get all countdown elements
        var countdownElements = document.querySelectorAll('.countdown');
        
        countdownElements.forEach(function(el) {
            var span = el.querySelector('span');
            // Get the target timestamp from data attribute
            var targetTime = parseInt(el.getAttribute('data-target'));
            
            // Calculate difference between now and target time
            var now = new Date().getTime();
            var diff = targetTime - now;
            
            if (diff <= 0) {
                span.textContent = 'Available now';
                
                // Show the refresh button when countdown reaches zero
                var refreshBtn = el.parentElement.querySelector('.refresh-btn');
                if (refreshBtn) {
                    refreshBtn.style.display = 'flex';
                }
            } else {
                var hours = Math.floor(diff / (1000 * 60 * 60));
                var minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                var seconds = Math.floor((diff % (1000 * 60)) / 1000);
                
                // Format with leading zeros
                hours = hours < 10 ? '0' + hours : hours;
                minutes = minutes < 10 ? '0' + minutes : minutes;
                seconds = seconds < 10 ? '0' + seconds : seconds;
                
                span.textContent = hours + ':' + minutes + ':' + seconds;
            }
        });
    }
    
    // Update countdowns immediately
    updateCountdowns();
    
    // Update countdowns every second
    setInterval(updateCountdowns, 1000);
});
</script>

<?php include "includes/footer.php"; ?>