<?php
// Initialize variables
 $ptitle = "Investment Packages";
 $username = $_SESSION['username'] ?? '';
 $userId = null; // Will be set dynamically

// Include required files
require $_SERVER['DOCUMENT_ROOT']."/stream.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/includes/generalinclude.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/user/includes/generalinclude.php";

// Set timezone
date_default_timezone_set('UTC');

// Fix database structure for transactions table
 $fixTransactionTable = $link->query("ALTER TABLE flow_transactions MODIFY COLUMN type VARCHAR(20)");
if ($fixTransactionTable === false) {
    // Log error but don't stop execution
    error_log("Error altering flow_transactions table: " . $link->error);
}

// Get user ID from session or database
// First, try to get user ID from session
if (isset($_SESSION['user_id'])) {
    $userId = $_SESSION['user_id'];
} 
// If not in session, try to get from username in session
elseif (isset($_SESSION['username'])) {
    $username = $_SESSION['username'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE username = ?");
    $getUserQuery->bind_param("s", $username);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}
// If still not found, try to get from email in session
elseif (isset($_SESSION['email'])) {
    $email = $_SESSION['email'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE email = ?");
    $getUserQuery->bind_param("s", $email);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}

// If user ID still not found, create a default user for testing
if (!$userId) {
    // Check if default user exists
    $checkUserQuery = $link->prepare("SELECT id FROM users WHERE username = 'Test User'");
    $checkUserQuery->execute();
    $checkResult = $checkUserQuery->get_result();
    
    if ($checkResult->num_rows == 0) {
        // Create default user
        $insertUserQuery = $link->prepare("INSERT INTO users (username, email, password, funds) VALUES (?, ?, ?, ?)");
        $defaultUsername = 'Test User';
        $defaultEmail = 'test@example.com';
        $defaultPassword = password_hash('password123', PASSWORD_DEFAULT);
        $defaultFunds = 0.00;
        $insertUserQuery->bind_param("sssd", $defaultUsername, $defaultEmail, $defaultPassword, $defaultFunds);
        $insertUserQuery->execute();
        $userId = $link->insert_id;
    } else {
        $userData = $checkResult->fetch_assoc();
        $userId = $userData['id'];
    }
    
    // Set user_id in session
    $_SESSION['user_id'] = $userId;
    $_SESSION['username'] = 'Test User';
    $_SESSION['email'] = 'test@example.com';
}

// Check if flow_user_packages table exists, create if not
 $tableCheck = $link->query("SHOW TABLES LIKE 'flow_user_packages'");
if ($tableCheck->num_rows == 0) {
    $createTable = "CREATE TABLE `flow_user_packages` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `user_id` int(11) NOT NULL,
        `package_id` int(11) NOT NULL,
        `start_date` date NOT NULL,
        `end_date` date NOT NULL,
        `total_earned` decimal(10,2) DEFAULT 0,
        `status` enum('active','completed','sold') DEFAULT 'active',
        `last_credit_date` datetime DEFAULT NULL,
        `accumulated_income` decimal(10,2) DEFAULT 0,
        `is_sold` tinyint(1) DEFAULT 0,
        PRIMARY KEY (`id`),
        FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
        FOREIGN KEY (`package_id`) REFERENCES `flow_packages`(`id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
    
    if ($link->query($createTable)) {
        $pkgMsg = '<div class="alert alert-info">User packages table created successfully!</div>';
    } else {
        die("Error creating user packages table: " . $link->error);
    }
} else {
    // Check if accumulated_income column exists
    $columnCheck = $link->query("SHOW COLUMNS FROM flow_user_packages LIKE 'accumulated_income'");
    if ($columnCheck->num_rows == 0) {
        // Add the column
        $alterTable = "ALTER TABLE `flow_user_packages` ADD COLUMN `accumulated_income` decimal(10,2) DEFAULT 0 AFTER `total_earned`";
        $link->query($alterTable);
    }
    
    // Check if is_sold column exists
    $columnCheck = $link->query("SHOW COLUMNS FROM flow_user_packages LIKE 'is_sold'");
    if ($columnCheck->num_rows == 0) {
        // Add the column
        $alterTable = "ALTER TABLE `flow_user_packages` ADD COLUMN `is_sold` tinyint(1) DEFAULT 0 AFTER `accumulated_income`";
        $link->query($alterTable);
    }
    
    // Check if last_credit_date column exists and is datetime
    $columnCheck = $link->query("SHOW COLUMNS FROM flow_user_packages LIKE 'last_credit_date'");
    if ($columnCheck->num_rows > 0) {
        $columnInfo = $columnCheck->fetch_assoc();
        if ($columnInfo['Type'] !== 'datetime') {
            // Alter column to datetime
            $alterTable = "ALTER TABLE `flow_user_packages` MODIFY COLUMN `last_credit_date` datetime DEFAULT NULL";
            $link->query($alterTable);
        }
    } else {
        // Add the column as datetime
        $alterTable = "ALTER TABLE `flow_user_packages` ADD COLUMN `last_credit_date` datetime DEFAULT NULL";
        $link->query($alterTable);
    }
}

// Update expired packages (but don't change status to completed, keep as active until sold)
 $updateExpired = $link->prepare("UPDATE flow_user_packages SET status = 'completed' WHERE end_date < CURDATE() AND status = 'active' AND is_sold = 0");
 $updateExpired->execute();

// Process automatic daily income accumulation (not crediting to balance)
 $creditQuery = $link->prepare("
   SELECT up.*, p.daily_income 
   FROM flow_user_packages up 
   JOIN flow_packages p ON up.package_id = p.id 
   WHERE up.status = 'active' AND up.is_sold = 0 AND 
   (up.last_credit_date IS NULL OR up.last_credit_date < DATE_SUB(NOW(), INTERVAL 24 HOUR))
");
 $creditQuery->execute();
 $creditResult = $creditQuery->get_result();

while ($package = $creditResult->fetch_assoc()) {
    $packageId = $package['id'];
    $dailyIncome = $package['daily_income'];
    
    // Update the package: set last_credit_date to now and add to accumulated_income
    $updatePackage = $link->prepare("UPDATE flow_user_packages SET last_credit_date = NOW(), accumulated_income = accumulated_income + ? WHERE id = ?");
    $updatePackage->bind_param("di", $dailyIncome, $packageId);
    $updatePackage->execute();
}

// Function to process referral bonus
function processReferralBonus($userId, $packageId) {
    global $link;
    
    try {
        // Get the user's referrer
        $stmt = $link->prepare("SELECT referral_id FROM users WHERE id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $user = $result->fetch_assoc();
            $referrerId = $user['referral_id'];
            
            if ($referrerId > 0) {
                // Get the package details
                $pkgStmt = $link->prepare("SELECT referral_bonus FROM flow_packages WHERE id = ?");
                $pkgStmt->bind_param("i", $packageId);
                $pkgStmt->execute();
                $pkgResult = $pkgStmt->get_result();
                
                if ($pkgResult->num_rows > 0) {
                    $package = $pkgResult->fetch_assoc();
                    $bonusAmount = $package['referral_bonus'];
                    
                    if ($bonusAmount > 0) {
                        // Check if this bonus has already been processed
                        $checkStmt = $link->prepare("SELECT id FROM flow_referral_bonuses 
                                                   WHERE user_id = ? AND referred_user_id = ? AND package_id = ?");
                        $checkStmt->bind_param("iii", $referrerId, $userId, $packageId);
                        $checkStmt->execute();
                        $checkResult = $checkStmt->get_result();
                        
                        if ($checkResult->num_rows == 0) {
                            // Record the bonus
                            $insertStmt = $link->prepare("INSERT INTO flow_referral_bonuses 
                                                        (user_id, referred_user_id, package_id, amount, status) 
                                                        VALUES (?, ?, ?, ?, 'paid')");
                            $insertStmt->bind_param("iiid", $referrerId, $userId, $packageId, $bonusAmount);
                            
                            if ($insertStmt->execute()) {
                                // Update the user's referral funds
                                $updateStmt = $link->prepare("UPDATE users SET referral_funds = referral_funds + ? WHERE id = ?");
                                $updateStmt->bind_param("di", $bonusAmount, $referrerId);
                                $updateStmt->execute();
                                
                                return true;
                            }
                        }
                    }
                }
            }
        }
        return false;
    } catch (Exception $e) {
        error_log("Referral bonus processing error: " . $e->getMessage());
        return false;
    }
}

// Handle package subscription (buy)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['subscribePackage'])) {
    // Check if this is a form resubmission
    if (isset($_POST['form_token']) && isset($_SESSION['last_form_token']) && $_POST['form_token'] === $_SESSION['last_form_token']) {
        // This is a form resubmission, ignore it
        unset($_SESSION['last_form_token']);
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    $pkgId = $_POST['pkgId'];
    
    // Get package details
    $pkgQuery = $link->prepare("SELECT * FROM flow_packages WHERE id = ?");
    $pkgQuery->bind_param("i", $pkgId);
    $pkgQuery->execute();
    $pkgResult = $pkgQuery->get_result();
    
    if ($pkgResult->num_rows > 0) {
        $package = $pkgResult->fetch_assoc();
        $amount = $package['amount'];
        $duration = $package['duration'];
        $packageName = $package['name'];
        $totalIncome = $package['total_income'];
        
        // Check user balance
        $balanceQuery = $link->prepare("SELECT funds FROM users WHERE id = ?");
        $balanceQuery->bind_param("i", $userId);
        $balanceQuery->execute();
        $balanceResult = $balanceQuery->get_result();
        $user = $balanceResult->fetch_assoc();
        $balance = $user['funds'];
        
        if ($balance >= $amount) {
            // Deduct amount from user balance
            $updateBalance = $link->prepare("UPDATE users SET funds = funds - ? WHERE id = ?");
            $updateBalance->bind_param("di", $amount, $userId);
            $updateBalance->execute();
            
            // Calculate dates
            $startDate = date('Y-m-d');
            $endDate = date('Y-m-d', strtotime("+$duration days"));
            
            // Insert subscription
            $insertSub = $link->prepare("INSERT INTO flow_user_packages (user_id, package_id, start_date, end_date, accumulated_income, status) VALUES (?, ?, ?, ?, 0, 'active')");
            $insertSub->bind_param("iiss", $userId, $pkgId, $startDate, $endDate);
            $insertSub->execute();
            
            // Record transaction
            $transactionId = 'PKG' . time() . rand(1000, 9999);
            $description = "Package purchase: " . $packageName;
            $insertTransaction = $link->prepare("INSERT INTO flow_transactions (user_id, transaction_id, amount, type, description, created_at) VALUES (?, ?, ?, 'purchase', ?, NOW())");
            $insertTransaction->bind_param("idss", $userId, $transactionId, $amount, $description);
            $insertTransaction->execute();
            
            // Process referral bonus
            processReferralBonus($userId, $pkgId);
            
            // Store form token in session to prevent resubmission
            $_SESSION['last_form_token'] = $_POST['form_token'] ?? bin2hex(random_bytes(16));
            
            // Calculate profit
            $profit = $totalIncome - $amount;
            
            // Create success message with package details
            $_SESSION['success_message'] = [
                'type' => 'purchase',
                'title' => 'Package Purchased Successfully!',
                'details' => [
                    'Package Name' => $packageName,
                    'Amount Paid' => '₦' . number_format($amount, 2),
                    'Duration' => $duration . ' days',
                    'Total Return' => '₦' . number_format($totalIncome, 2),
                    'Expected Profit' => '₦' . number_format($profit, 2)
                ]
            ];
            
            // Redirect to prevent form resubmission
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        } else {
            $_SESSION['error_message'] = 'Insufficient balance! You need ₦' . number_format($amount, 2) . ' but have only ₦' . number_format($balance, 2);
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        }
    } else {
        $_SESSION['error_message'] = 'Invalid package selected!';
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
}

// Handle package selling
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['sellPackage'])) {
    // Check if this is a form resubmission
    if (isset($_POST['form_token']) && isset($_SESSION['last_sell_form_token']) && $_POST['form_token'] === $_SESSION['last_sell_form_token']) {
        // This is a form resubmission, ignore it
        unset($_SESSION['last_sell_form_token']);
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    $userPackageId = $_POST['userPackageId'];
    
    // Get package details
    $pkgQuery = $link->prepare("
        SELECT up.*, p.name, p.amount 
        FROM flow_user_packages up 
        JOIN flow_packages p ON up.package_id = p.id 
        WHERE up.id = ? AND up.user_id = ? AND up.is_sold = 0
    ");
    $pkgQuery->bind_param("ii", $userPackageId, $userId);
    $pkgQuery->execute();
    $pkgResult = $pkgQuery->get_result();
    
    if ($pkgResult->num_rows > 0) {
        $package = $pkgResult->fetch_assoc();
        $accumulatedIncome = $package['accumulated_income'];
        $packageAmount = $package['amount'];
        $packageName = $package['name'];
        
        // Check if package has expired (end date is today or in the past)
        $today = new DateTime();
        $endDate = new DateTime($package['end_date']);
        
        if ($endDate <= $today) {
            // Add accumulated income to user's funds
            $updateFunds = $link->prepare("UPDATE users SET funds = funds + ? WHERE id = ?");
            $updateFunds->bind_param("di", $accumulatedIncome, $userId);
            $updateFunds->execute();
            
            // Update the package as sold
            $updatePackage = $link->prepare("UPDATE flow_user_packages SET is_sold = 1, status = 'sold', total_earned = ? WHERE id = ?");
            $updatePackage->bind_param("di", $accumulatedIncome, $userPackageId);
            $updatePackage->execute();
            
            // Record transaction
            $transactionId = 'SELL' . time() . rand(1000, 9999);
            $description = "Package sale: " . $packageName;
            $insertTransaction = $link->prepare("INSERT INTO flow_transactions (user_id, transaction_id, amount, type, description, created_at) VALUES (?, ?, ?, 'sale', ?, NOW())");
            $insertTransaction->bind_param("idss", $userId, $transactionId, $accumulatedIncome, $description);
            $insertTransaction->execute();
            
            // Store form token in session to prevent resubmission
            $_SESSION['last_sell_form_token'] = $_POST['form_token'] ?? bin2hex(random_bytes(16));
            
            // Calculate profit
            $profit = $accumulatedIncome - $packageAmount;
            
            // Create success message with package details
            $_SESSION['success_message'] = [
                'type' => 'sale',
                'title' => 'Package Sold Successfully!',
                'details' => [
                    'Package Name' => $packageName,
                    'Initial Investment' => '₦' . number_format($packageAmount, 2),
                    'Amount Earned' => '₦' . number_format($accumulatedIncome, 2),
                    'Net Profit' => '₦' . number_format($profit, 2)
                ]
            ];
            
            // Redirect to prevent form resubmission
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        } else {
            $_SESSION['error_message'] = 'Package has not expired yet! You can only sell after the package duration ends.';
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        }
    } else {
        $_SESSION['error_message'] = 'Invalid package or already sold!';
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
}

// Get user's active packages (subscribed packages that are still active)
 $activeQuery = $link->prepare("
   SELECT up.*, p.name, p.amount, p.daily_income, p.total_income, p.min_withdrawal 
   FROM flow_user_packages up 
   JOIN flow_packages p ON up.package_id = p.id 
   WHERE up.user_id = ? AND up.status = 'active' AND up.is_sold = 0
   ORDER BY up.id DESC
");
 $activeQuery->bind_param("i", $userId);
 $activeQuery->execute();
 $activeResult = $activeQuery->get_result();

// Get user's completed packages (expired but not sold)
 $completedQuery = $link->prepare("
   SELECT up.*, p.name, p.amount, p.daily_income, p.total_income 
   FROM flow_user_packages up 
   JOIN flow_packages p ON up.package_id = p.id 
   WHERE up.user_id = ? AND up.status = 'completed' AND up.is_sold = 0
   ORDER BY up.id DESC
");
 $completedQuery->bind_param("i", $userId);
 $completedQuery->execute();
 $completedResult = $completedQuery->get_result();

// Get user's sold packages
 $soldQuery = $link->prepare("
   SELECT up.*, p.name, p.amount, p.daily_income, p.total_income 
   FROM flow_user_packages up 
   JOIN flow_packages p ON up.package_id = p.id 
   WHERE up.user_id = ? AND up.is_sold = 1
   ORDER BY up.id DESC
");
 $soldQuery->bind_param("i", $userId);
 $soldQuery->execute();
 $soldResult = $soldQuery->get_result();

// Get all available packages (packages that user hasn't subscribed to at all)
 $availableQuery = $link->prepare("
   SELECT p.* 
   FROM flow_packages p
   WHERE p.id NOT IN (
       SELECT package_id FROM flow_user_packages WHERE user_id = ?
   )
   ORDER BY p.id DESC
");
 $availableQuery->bind_param("i", $userId);
 $availableQuery->execute();
 $availableResult = $availableQuery->get_result();

// Check if any tabs should be hidden
 $hasActive = $activeResult->num_rows > 0;
 $hasCompleted = $completedResult->num_rows > 0;
 $hasSold = $soldResult->num_rows > 0;
 $hasAvailable = $availableResult->num_rows > 0;

// Get success and error messages from session
 $successMessage = isset($_SESSION['success_message']) ? $_SESSION['success_message'] : null;
 $errorMessage = isset($_SESSION['error_message']) ? $_SESSION['error_message'] : null;

// Clear messages from session
unset($_SESSION['success_message']);
unset($_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <meta name="description" content="<?php echo $sitedesc ?>">
    <title><?php echo $ptitle ?> - <?php echo $sitename ?></title>
    <link rel="shortcut icon" href="../favicon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css"/>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css"/>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        :root {
            /* Gainmart colors */
            --primary-blue: #75AFEE;
            --dark-blue: #4A8FE7;
            --darker-blue: #2E6CBF;
            --black: #000000;
            --dark-gray: #1A1A1A;
            --medium-gray: #2A2A2A;
            --light-gray: #3A3A3A;
            --text-light: #E0E0E0;
            --text-dim: #A0A0A0;
            --card-bg: #1F1F1F;
            --card-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
            --input-bg: #2A2A2A;
            --border-color: #3A3A3A;
            --success: #4CAF50;
            --warning: #FFC107;
            --danger: #F44336;
            --buy-color: #4CAF50; /* Green */
            --sell-color: #F44336; /* Red */
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--black);
            color: var(--text-light);
            transition: all 0.3s ease;
            min-height: 100vh;
            padding-bottom: 5rem; /* Added padding for bottom nav */
        }

        h1,h2,h3,h4,h5,h6 {
            font-weight: 700;
        }
        
        .container-fluid {
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
            margin-right: auto;
            margin-left: auto;
        }

        .row {
            display: flex;
            flex-wrap: wrap;
            margin-right: -15px;
            margin-left: -15px;
        }

        .col-12 {
            position: relative;
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
        }

        .g-gs {
            --bs-gutter-x: 1.5rem;
            --bs-gutter-y: 0;
            display: flex;
            flex-wrap: wrap;
            margin-top: calc(-1 * var(--bs-gutter-y));
            margin-right: calc(-.5 * var(--bs-gutter-x));
            margin-left: calc(-.5 * var(--bs-gutter-x));
        }

        .g-gs > * {
            flex-shrink: 0;
            width: 100%;
            max-width: 100%;
            padding-right: calc(var(--bs-gutter-x) * .5);
            padding-left: calc(var(--bs-gutter-x) * .5);
            margin-top: var(--bs-gutter-y);
        }

        .mt-3 {
            margin-top: 1rem !important;
        }

        .mt-4 {
            margin-top: 1.5rem !important;
        }

        .px-3 {
            padding-left: 1rem !important;
            padding-right: 1rem !important;
        }

        .mb-3 {
            margin-bottom: 1rem !important;
        }

        .mr-2 {
            margin-right: 0.5rem !important;
        }

        /* Header Styles */
        .header {
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            padding: 1rem 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .logo img {
            height: 40px;
        }

        .logo-text {
            font-size: 1.5rem;
            font-weight: 800;
            color: var(--primary-blue);
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .notification-btn {
            position: relative;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.2rem;
            cursor: pointer;
            padding: 0.75rem;
            border-radius: 50%;
            transition: all 0.3s ease;
        }

        .notification-btn:hover {
            background: var(--light-gray);
            color: var(--primary-blue);
        }

        .notification-badge {
            position: absolute;
            top: 0;
            right: 0;
            width: 10px;
            height: 10px;
            background: var(--danger);
            border-radius: 50%;
        }

        .account-section {
            position: relative;
        }

        .account-btn {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            cursor: pointer;
            padding: 0.5rem 1rem;
            border-radius: 2rem;
            transition: all 0.3s ease;
        }

        .account-btn:hover {
            background: var(--light-gray);
        }

        .account-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--primary-blue);
        }

        .account-info {
            display: flex;
            flex-direction: column;
            align-items: flex-start;
        }

        .account-name {
            font-weight: 600;
            font-size: 0.9rem;
        }

        .account-username {
            font-size: 0.8rem;
            color: var(--text-dim);
        }

        .account-dropdown {
            position: absolute;
            top: 100%;
            right: 0;
            margin-top: 0.5rem;
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            border-radius: 0.75rem;
            box-shadow: 0 10px 25px rgba(0,0,0,0.4);
            min-width: 220px;
            overflow: hidden;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.3s ease;
        }

        .account-dropdown.active {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }

        .dropdown-divider {
            height: 1px;
            background: var(--light-gray);
            margin: 0.5rem 0;
        }

        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: var(--text-light);
            text-decoration: none;
            transition: background-color 0.3s;
        }

        .dropdown-item:hover {
            background-color: var(--medium-gray);
        }

        .dropdown-item i {
            font-size: 1.1rem;
            width: 20px;
            text-align: center;
            color: var(--primary-blue);
        }

        /* Bottom Navigation Bar */
        .bottom-navigation {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: var(--dark-gray);
            box-shadow: 0 -2px 10px rgba(0,0,0,0.2);
            z-index: 100;
            padding: 0.75rem 0;
            display: flex;
            justify-content: space-around;
            align-items: center;
        }

        .bottom-nav-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
            color: var(--text-dim);
            text-decoration: none;
            width: 20%;
        }

        .bottom-nav-item:hover {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-item.active {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-icon {
            font-size: 1.5rem;
            margin-bottom: 0.25rem;
        }

        .bottom-nav-text {
            font-size: 0.7rem;
            font-weight: 600;
        }

        /* Right Sidebar Menu */
        .right-sidebar {
            position: fixed;
            top: 0;
            right: -50%; /* Hidden by default - half page width */
            width: 50%; /* Half page width */
            height: 100%;
            background: var(--dark-gray);
            z-index: 1001;
            transition: right 0.4s cubic-bezier(0.25, 0.8, 0.25, 1);
            overflow-y: auto;
            box-shadow: -5px 0 25px rgba(0,0,0,0.4);
        }

        .right-sidebar.active {
            right: 0;
        }

        .sidebar-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem;
            border-bottom: 1px solid var(--light-gray);
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
        }

        .sidebar-logo {
            height: 40px;
            filter: drop-shadow(0 2px 4px rgba(0,0,0,0.3));
        }

        .sidebar-close {
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
        }

        .sidebar-close:hover {
            background: var(--light-gray);
            transform: rotate(90deg);
        }

        .sidebar-content {
            padding: 1rem 0;
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
            min-height: calc(100% - 73px); /* Subtract header height */
        }

        .nk-menu {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .nk-menu-item {
            margin: 0.25rem 0;
            opacity: 0;
            transform: translateX(20px);
            animation: slideInRight 0.5s forwards;
        }

        /* Stagger animation for menu items */
        .nk-menu-item:nth-child(1) { animation-delay: 0.1s; }
        .nk-menu-item:nth-child(2) { animation-delay: 0.15s; }
        .nk-menu-item:nth-child(3) { animation-delay: 0.2s; }
        .nk-menu-item:nth-child(4) { animation-delay: 0.25s; }
        .nk-menu-item:nth-child(5) { animation-delay: 0.3s; }
        .nk-menu-item:nth-child(6) { animation-delay: 0.35s; }
        .nk-menu-item:nth-child(7) { animation-delay: 0.4s; }
        .nk-menu-item:nth-child(8) { animation-delay: 0.45s; }
        .nk-menu-item:nth-child(9) { animation-delay: 0.5s; }
        .nk-menu-item:nth-child(10) { animation-delay: 0.55s; }
        .nk-menu-item:nth-child(11) { animation-delay: 0.6s; }
        .nk-menu-item:nth-child(12) { animation-delay: 0.65s; }
        .nk-menu-item:nth-child(13) { animation-delay: 0.7s; }
        .nk-menu-item:nth-child(14) { animation-delay: 0.75s; }
        .nk-menu-item:nth-child(15) { animation-delay: 0.8s; }

        @keyframes slideInRight {
            from {
                opacity: 0;
                transform: translateX(20px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }

        .nk-menu-link {
            display: flex;
            align-items: center;
            padding: 0.9rem 1.5rem;
            color: var(--text-light);
            text-decoration: none;
            transition: all 0.3s ease;
            border-radius: 0.75rem;
            margin: 0 0.75rem;
            position: relative;
            overflow: hidden;
        }

        .nk-menu-link::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary-blue);
            transform: scaleY(0);
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover {
            background: var(--medium-gray);
            transform: translateX(5px);
        }

        .nk-menu-link:hover::before {
            transform: scaleY(1);
        }

        .nk-menu-icon {
            margin-right: 1rem;
            color: var(--primary-blue);
            font-size: 1.2rem;
            width: 24px;
            text-align: center;
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover .nk-menu-icon {
            transform: scale(1.1);
        }

        .nk-menu-text {
            font-weight: 500;
            font-size: 1rem;
        }

        .nk-menu-heading {
            padding: 0.75rem 1.5rem;
            margin: 1.5rem 0 0.75rem;
            color: var(--text-dim);
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.1em;
            position: relative;
        }

        .nk-menu-heading::after {
            content: '';
            position: absolute;
            bottom: -0.5rem;
            left: 1.5rem;
            right: 1.5rem;
            height: 1px;
            background: var(--light-gray);
        }

        .overline-title {
            font-weight: 600;
        }

        .nk-menu-sub {
            list-style: none;
            padding: 0;
            margin: 0.25rem 0 0.25rem 2rem;
            background: var(--black);
            border-radius: 0.75rem;
            overflow: hidden;
            max-height: 0;
            opacity: 0;
            transition: all 0.4s ease;
        }

        .nk-menu-sub.active {
            max-height: 500px;
            opacity: 1;
        }

        .nk-menu-sub .nk-menu-item {
            margin: 0;
            opacity: 1;
            transform: none;
            animation: none;
        }

        .nk-menu-sub .nk-menu-link {
            padding: 0.7rem 1rem 0.7rem 1.5rem;
            font-size: 0.9rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
        }

        .nk-menu-sub .nk-menu-link::before {
            width: 3px;
        }

        .nk-menu-sub .nk-menu-icon {
            font-size: 1rem;
            margin-right: 0.75rem;
        }

        /* Menu toggle icon animation */
        .nk-menu-toggle .nk-menu-icon::after {
            content: '\f107';
            font-family: 'Font Awesome 6 Free';
            font-weight: 900;
            margin-left: auto;
            transition: transform 0.3s ease;
        }

        .nk-menu-toggle.active .nk-menu-icon::after {
            transform: rotate(180deg);
        }

        /* Overlay for sidebar */
        .sidebar-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .sidebar-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        /* Page Title */
        .page-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1.5rem;
            padding: 0 1rem;
        }

        /* Package Tabs */
        .package-tabs {
            display: flex;
            overflow-x: auto;
            margin-bottom: 1.5rem;
            padding: 0 1rem;
            scrollbar-width: thin;
            scrollbar-color: var(--primary-blue) var(--border-color);
        }

        .package-tabs::-webkit-scrollbar {
            height: 6px;
        }

        .package-tabs::-webkit-scrollbar-track {
            background: var(--border-color);
            border-radius: 10px;
        }

        .package-tabs::-webkit-scrollbar-thumb {
            background-color: var(--primary-blue);
            border-radius: 10px;
        }

        .package-tab {
            flex: 0 0 auto;
            padding: 0.75rem 1.25rem;
            background: var(--medium-gray);
            color: var(--text-dim);
            font-weight: 600;
            font-size: 0.9rem;
            border-radius: 2rem;
            margin-right: 0.75rem;
            cursor: pointer;
            transition: all 0.3s ease;
            white-space: nowrap;
        }

        .package-tab.active {
            background: var(--primary-blue);
            color: var(--black);
        }

        .package-tab:hover:not(.active) {
            background: var(--light-gray);
            color: var(--text-light);
        }

        /* Package Cards */
        .packages-container {
            padding: 0 1rem;
        }

        .packages-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1.25rem;
        }

        .package-card {
            background: var(--dark-gray);
            border-radius: 0.75rem;
            padding: 1.25rem;
            box-shadow: var(--card-shadow);
            transition: transform 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .package-card:hover {
            transform: translateY(-5px);
        }

        .package-card::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 100px;
            height: 100px;
            background: var(--primary-blue);
            opacity: 0.1;
            border-radius: 50%;
            transform: translate(30px, -30px);
        }

        .package-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
            position: relative;
            z-index: 1;
        }

        .package-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: var(--text-light);
            margin: 0;
        }

        .package-amount {
            font-size: 1.5rem;
            font-weight: 800;
            color: var(--primary-blue);
            margin: 0.5rem 0;
        }

        .package-amount::before {
            content: "₦";
            margin-right: 0.25rem;
            font-size: 1rem;
        }

        .package-details {
            margin-bottom: 1.25rem;
            position: relative;
            z-index: 1;
        }

        .package-detail {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
            font-size: 0.85rem;
        }

        .package-detail-label {
            color: var(--text-dim);
        }

        .package-detail-value {
            font-weight: 600;
            color: var(--text-light);
        }

        .package-progress {
            margin: 1rem 0;
            position: relative;
            z-index: 1;
        }

        .progress-container {
            background-color: var(--medium-gray);
            height: 8px;
            border-radius: 10px;
            overflow: hidden;
            margin-bottom: 0.5rem;
        }

        .progress-bar {
            height: 100%;
            background-color: var(--primary-blue);
            border-radius: 10px;
            transition: width 1s ease-in-out;
        }

        .progress-text {
            display: flex;
            justify-content: space-between;
            font-size: 0.8rem;
            color: var(--text-dim);
        }

        .package-footer {
            margin-top: auto;
            position: relative;
            z-index: 1;
            display: flex;
            gap: 0.75rem;
        }

        .package-btn {
            flex: 1;
            padding: 0.75rem;
            border: none;
            border-radius: 0.5rem;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 600;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }

        .package-btn.buy {
            background: var(--buy-color);
            color: var(--black);
        }

        .package-btn.buy:hover {
            background: #3d8b40;
            transform: translateY(-2px);
        }

        .package-btn.sell {
            background: var(--sell-color);
            color: white;
        }

        .package-btn.sell:hover {
            background: #d32f2f;
            transform: translateY(-2px);
        }

        .package-btn:disabled {
            background: var(--light-gray);
            color: var(--text-dim);
            cursor: not-allowed;
            transform: none;
        }

        .countdown {
            font-size: 0.85rem;
            color: var(--primary-blue);
            text-align: center;
            margin: 0.75rem 0;
        }

        .auto-credit-note {
            font-size: 0.75rem;
            color: var(--text-dim);
            text-align: center;
            margin-top: 0.5rem;
            font-style: italic;
        }

        .completed-badge {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: var(--warning);
            color: var(--black);
            padding: 0.25rem 0.75rem;
            border-radius: 1rem;
            font-size: 0.75rem;
            font-weight: 700;
            z-index: 2;
        }

        .sold-badge {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: var(--success);
            color: var(--black);
            padding: 0.25rem 0.75rem;
            border-radius: 1rem;
            font-size: 0.75rem;
            font-weight: 700;
            z-index: 2;
        }

        .empty-state {
            text-align: center;
            padding: 3rem 1rem;
            color: var(--text-dim);
        }

        .empty-state-icon {
            font-size: 3rem;
            color: var(--light-gray);
            margin-bottom: 1rem;
        }

        .empty-state-text {
            font-size: 1rem;
            margin-bottom: 1.5rem;
        }

        /* Success and Error Modal */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        .modal {
            background: var(--dark-gray);
            border-radius: 1rem;
            width: 90%;
            max-width: 500px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.5);
            transform: scale(0.8);
            transition: transform 0.3s ease;
            overflow: hidden;
        }

        .modal-overlay.active .modal {
            transform: scale(1);
        }

        .modal-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--light-gray);
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .modal-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
        }

        .modal-icon.success {
            background: var(--success);
            color: var(--black);
        }

        .modal-icon.error {
            background: var(--danger);
            color: white;
        }

        .modal-title {
            font-size: 1.2rem;
            font-weight: 700;
        }

        .modal-body {
            padding: 1.5rem;
        }

        .modal-details {
            margin-bottom: 1.5rem;
        }

        .modal-detail {
            display: flex;
            justify-content: space-between;
            padding: 0.75rem 0;
            border-bottom: 1px solid var(--light-gray);
        }

        .modal-detail:last-child {
            border-bottom: none;
        }

        .modal-detail-label {
            color: var(--text-dim);
            font-weight: 500;
        }

        .modal-detail-value {
            font-weight: 600;
            color: var(--text-light);
        }

        .modal-footer {
            padding: 1.5rem;
            border-top: 1px solid var(--light-gray);
            display: flex;
            justify-content: flex-end;
        }

        .modal-btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 0.5rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .modal-btn-primary {
            background: var(--primary-blue);
            color: var(--black);
        }

        .modal-btn-primary:hover {
            background: var(--dark-blue);
        }

        /* Copy notification */
        .copy-notification {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            background-color: var(--primary-blue);
            color: var(--black);
            padding: 12px 24px;
            border-radius: 8px;
            z-index: 9999;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            opacity: 1;
            transition: opacity 0.5s;
            font-weight: 600;
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .header {
                padding: 0.75rem 1rem;
            }
            
            .logo-text {
                font-size: 1.2rem;
            }
            
            .account-info {
                display: none;
            }
            
            .right-sidebar {
                width: 80%; /* 80% width on mobile instead of 50% */
                right: -80%;
            }
            
            .packages-grid {
                grid-template-columns: 1fr;
            }
            
            .bottom-nav-text {
                font-size: 0.6rem;
            }
            
            .package-footer {
                flex-direction: column;
            }
            
            .modal {
                width: 95%;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="logo">
            <img src="/g1.jpg" alt="Gainmart Logo">
            <div class="logo-text">Gainmart</div>
        </div>
        <div class="header-actions">
            <div class="account-section">
                <button class="account-btn" id="accountBtn">
                    <img src="/g1.jpg" alt="Profile" class="account-avatar">
                    <div class="account-info">
                        <span class="account-name"><?php echo $fullname ?></span>
                        <span class="account-username">@<?php echo $username ?></span>
                    </div>
                    <i class="fas fa-chevron-down"></i>
                </button>
                
                <div class="account-dropdown" id="accountDropdown">
                    <a href="settings" class="dropdown-item">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                    <div class="dropdown-divider"></div>
                    <a href="logout" class="dropdown-item">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Log Out</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Right Sidebar Menu -->
    <div class="right-sidebar" id="rightSidebar">
        <div class="sidebar-header">
            <img src="favicon.png" alt="Logo" class="sidebar-logo">
            <button class="sidebar-close" id="sidebarClose">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="sidebar-content">
            <div class="nk-sidebar-menu" data-simplebar>
                <ul class="nk-menu">
                    <li class="nk-menu-item">
                        <a href="../user/index" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-home"></i></span>
                            <span class="nk-menu-text">Dashboard</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_packages.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-box"></i></span>
                            <span class="nk-menu-text">Investment Packages</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_withdrawal.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-money-bill-wave"></i></span>
                            <span class="nk-menu-text">Withdraw Funds</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="referrals" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-users"></i></span>
                            <span class="nk-menu-text">My Referrals</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="settings" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-cog"></i></span>
                            <span class="nk-menu-text">Account Settings</span>
                        </a>
                    </li>
                    
                    <div class="mt-5"></div>
                </ul>
            </div>
        </div>
    </div>

    <!-- Sidebar Overlay -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <div class="nk-content px-0">
        <div class="container-fluid">
            <div class="nk-content-inner">
                <div class="nk-content-body">
                    <div class="row g-gs">

                        <div class="col-12 px-3 mt-4">
                            <h1 class="page-title">Investment Packages</h1>
                        </div>
                        
                        <!-- Package Tabs - Only show if there are packages in that category -->
                        <div class="col-12">
                            <div class="package-tabs">
                                <?php if ($hasActive): ?>
                                    <div class="package-tab active" data-tab="active">Active Packages</div>
                                <?php endif; ?>
                                
                                <?php if ($hasCompleted): ?>
                                    <div class="package-tab <?= !$hasActive ? 'active' : '' ?>" data-tab="completed">Completed Packages</div>
                                <?php endif; ?>
                                
                                <?php if ($hasSold): ?>
                                    <div class="package-tab <?= (!$hasActive && !$hasCompleted) ? 'active' : '' ?>" data-tab="sold">Sold Packages</div>
                                <?php endif; ?>
                                
                                <?php if ($hasAvailable): ?>
                                    <div class="package-tab <?= (!$hasActive && !$hasCompleted && !$hasSold) ? 'active' : '' ?>" data-tab="available">Available Packages</div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Active Packages - Only show if there are active packages -->
                        <?php if ($hasActive): ?>
                            <div class="col-12 package-content <?= $hasActive ? 'active' : '' ?>" id="active-packages">
                                <div class="packages-container">
                                    <div class="packages-grid">
                                        <?php 
                                            // Reset the result pointer to fetch data again
                                            $activeResult->data_seek(0);
                                            while ($pkg = $activeResult->fetch_assoc()): 
                                        ?>
                                            <?php
                                                $id = $pkg['id'];
                                                $name = $pkg['name'];
                                                $amount = $pkg['amount'];
                                                $daily_income = $pkg['daily_income'];
                                                $total_income = $pkg['total_income'];
                                                $start_date = $pkg['start_date'];
                                                $end_date = $pkg['end_date'];
                                                $accumulated_income = $pkg['accumulated_income'];
                                                $last_credit_date = $pkg['last_credit_date'];
                                                
                                                // Calculate days left and progress
                                                $today = new DateTime();
                                                $end = new DateTime($end_date);
                                                $start = new DateTime($start_date);
                                                $days_total = $start->diff($end)->days;
                                                $days_left = $today->diff($end)->days;
                                                $days_left = $days_left > 0 ? $days_left : 0;
                                                $progress = (($days_total - $days_left) / $days_total) * 100;
                                                
                                                // Calculate expected earnings
                                                $expected_earnings = $daily_income * ($days_total - $days_left);
                                                
                                                // Calculate next credit time
                                                if ($last_credit_date === null) {
                                                    // First credit: 24 hours after start date
                                                    $nextCreditTime = new DateTime($start_date);
                                                    $nextCreditTime->modify('+24 hours');
                                                } else {
                                                    // Subsequent credits: 24 hours after last credit
                                                    $nextCreditTime = new DateTime($last_credit_date);
                                                    $nextCreditTime->modify('+24 hours');
                                                }
                                                $nextCreditTimestamp = $nextCreditTime->getTimestamp();
                                                
                                                // Check if package is ready to sell (expired)
                                                $isReadyToSell = ($end <= $today);
                                            ?>
                                            <div class="package-card">
                                                <div class="package-header">
                                                    <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                </div>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                                <div class="package-details">
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Daily Income:</span>
                                                        <span class="package-detail-value">₦<?= number_format($daily_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Total Return:</span>
                                                        <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Accumulated Income:</span>
                                                        <span class="package-detail-value">₦<?= number_format($accumulated_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Expected Earnings:</span>
                                                        <span class="package-detail-value">₦<?= number_format($expected_earnings, 2) ?></span>
                                                    </div>
                                                </div>
                                                
                                                <div class="package-progress">
                                                    <div class="progress-container">
                                                        <div class="progress-bar" style="width: <?= $progress ?>%"></div>
                                                    </div>
                                                    <div class="progress-text">
                                                        <span><?= $days_left ?> days left</span>
                                                        <span><?= round($progress) ?>%</span>
                                                    </div>
                                                </div>
                                                
                                                <div class="countdown" id="countdown-<?= $id ?>" data-target="<?= $nextCreditTimestamp ?>">
                                                    Next credit in: <span>24:00:00</span>
                                                </div>
                                                <div class="auto-credit-note">
                                                    Your daily income is accumulating and will be available when you sell the package after expiration.
                                                </div>
                                                
                                                <div class="package-footer">
                                                    <form method="POST">
                                                        <input type="hidden" name="userPackageId" value="<?= $id ?>">
                                                        <input type="hidden" name="form_token" value="<?= bin2hex(random_bytes(16)) ?>">
                                                        <button type="submit" name="sellPackage" class="package-btn sell" <?= !$isReadyToSell ? 'disabled' : '' ?>>
                                                            <i class="fas fa-hand-holding-usd"></i>
                                                            Sell
                                                        </button>
                                                    </form>
                                                </div>
                                            </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Completed Packages - Only show if there are completed packages -->
                        <?php if ($hasCompleted): ?>
                            <div class="col-12 package-content <?= !$hasActive && $hasCompleted ? 'active' : '' ?>" id="completed-packages">
                                <div class="packages-container">
                                    <div class="packages-grid">
                                        <?php 
                                            // Reset the result pointer to fetch data again
                                            $completedResult->data_seek(0);
                                            while ($pkg = $completedResult->fetch_assoc()): 
                                        ?>
                                            <?php
                                                $id = $pkg['id'];
                                                $name = $pkg['name'];
                                                $amount = $pkg['amount'];
                                                $total_income = $pkg['total_income'];
                                                $accumulated_income = $pkg['accumulated_income'];
                                                $start_date = $pkg['start_date'];
                                                $end_date = $pkg['end_date'];
                                                $package_id = $pkg['package_id'];
                                                
                                                // Calculate profit
                                                $profit = $accumulated_income - $amount;
                                            ?>
                                            <div class="package-card">
                                                <div class="completed-badge">READY TO SELL</div>
                                                <div class="package-header">
                                                    <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                </div>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                                <div class="package-details">
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Total Return:</span>
                                                        <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Accumulated Income:</span>
                                                        <span class="package-detail-value">₦<?= number_format($accumulated_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Net Profit:</span>
                                                        <span class="package-detail-value">₦<?= number_format($profit, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Duration:</span>
                                                        <span class="package-detail-value"><?= date('M j, Y', strtotime($start_date)) ?> - <?= date('M j, Y', strtotime($end_date)) ?></span>
                                                    </div>
                                                </div>
                                                <div class="package-footer">
                                                    <form method="POST">
                                                        <input type="hidden" name="userPackageId" value="<?= $id ?>">
                                                        <input type="hidden" name="form_token" value="<?= bin2hex(random_bytes(16)) ?>">
                                                        <button type="submit" name="sellPackage" class="package-btn sell">
                                                            <i class="fas fa-hand-holding-usd"></i>
                                                            Sell Package
                                                        </button>
                                                    </form>
                                                </div>
                                            </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Sold Packages - Only show if there are sold packages -->
                        <?php if ($hasSold): ?>
                            <div class="col-12 package-content <?= (!$hasActive && !$hasCompleted && $hasSold) ? 'active' : '' ?>" id="sold-packages">
                                <div class="packages-container">
                                    <div class="packages-grid">
                                        <?php 
                                            // Reset the result pointer to fetch data again
                                            $soldResult->data_seek(0);
                                            while ($pkg = $soldResult->fetch_assoc()): 
                                        ?>
                                            <?php
                                                $id = $pkg['id'];
                                                $name = $pkg['name'];
                                                $amount = $pkg['amount'];
                                                $total_income = $pkg['total_income'];
                                                $total_earned = $pkg['total_earned'];
                                                $start_date = $pkg['start_date'];
                                                $end_date = $pkg['end_date'];
                                                $package_id = $pkg['package_id'];
                                                
                                                // Calculate profit
                                                $profit = $total_earned - $amount;
                                            ?>
                                            <div class="package-card">
                                                <div class="sold-badge">SOLD</div>
                                                <div class="package-header">
                                                    <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                </div>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                                <div class="package-details">
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Total Return:</span>
                                                        <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Amount Earned:</span>
                                                        <span class="package-detail-value">₦<?= number_format($total_earned, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Net Profit:</span>
                                                        <span class="package-detail-value">₦<?= number_format($profit, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Duration:</span>
                                                        <span class="package-detail-value"><?= date('M j, Y', strtotime($start_date)) ?> - <?= date('M j, Y', strtotime($end_date)) ?></span>
                                                    </div>
                                                </div>
                                                <div class="package-footer">
                                                    <form method="POST">
                                                        <input type="hidden" name="pkgId" value="<?= $package_id ?>">
                                                        <input type="hidden" name="form_token" value="<?= bin2hex(random_bytes(16)) ?>">
                                                        <button type="submit" name="subscribePackage" class="package-btn buy">
                                                            <i class="fas fa-shopping-cart"></i>
                                                            Buy Again
                                                        </button>
                                                    </form>
                                                </div>
                                            </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Available Packages - Only show if there are available packages -->
                        <?php if ($hasAvailable): ?>
                            <div class="col-12 package-content <?= (!$hasActive && !$hasCompleted && !$hasSold && $hasAvailable) ? 'active' : '' ?>" id="available-packages">
                                <div class="packages-container">
                                    <div class="packages-grid">
                                        <?php 
                                            // Reset the result pointer to fetch data again
                                            $availableResult->data_seek(0);
                                            while ($package = $availableResult->fetch_assoc()): 
                                        ?>
                                            <?php
                                                $id = $package['id'];
                                                $name = $package['name'];
                                                $amount = $package['amount'];
                                                $daily_income = $package['daily_income'];
                                                $duration = $package['duration'];
                                                $total_income = $package['total_income'];
                                                $min_withdrawal = $package['min_withdrawal'];
                                                $referral_bonus = $package['referral_bonus'];
                                                
                                                // Calculate ROI
                                                $roi = (($total_income - $amount) / $amount) * 100;
                                            ?>
                                            <div class="package-card">
                                                <div class="package-header">
                                                    <h5 class="package-title"><?= htmlspecialchars($name) ?></h5>
                                                </div>
                                                <div class="package-amount"><?= number_format($amount, 2) ?></div>
                                                <div class="package-details">
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Daily Income:</span>
                                                        <span class="package-detail-value">₦<?= number_format($daily_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Duration:</span>
                                                        <span class="package-detail-value"><?= $duration ?> days</span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Total Return:</span>
                                                        <span class="package-detail-value">₦<?= number_format($total_income, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">ROI:</span>
                                                        <span class="package-detail-value"><?= number_format($roi, 1) ?>%</span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Min. Withdrawal:</span>
                                                        <span class="package-detail-value">₦<?= number_format($min_withdrawal, 2) ?></span>
                                                    </div>
                                                    <div class="package-detail">
                                                        <span class="package-detail-label">Referral Bonus:</span>
                                                        <span class="package-detail-value">₦<?= number_format($referral_bonus, 2) ?></span>
                                                    </div>
                                                </div>
                                                <div class="package-footer">
                                                    <form method="POST">
                                                        <input type="hidden" name="pkgId" value="<?= $id ?>">
                                                        <input type="hidden" name="form_token" value="<?= bin2hex(random_bytes(16)) ?>">
                                                        <button type="submit" name="subscribePackage" class="package-btn buy">
                                                            <i class="fas fa-shopping-cart"></i>
                                                            Buy Now
                                                        </button>
                                                    </form>
                                                </div>
                                            </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Show empty state if no packages at all -->
                        <?php if (!$hasActive && !$hasCompleted && !$hasSold && !$hasAvailable): ?>
                            <div class="col-12">
                                <div class="empty-state">
                                    <div class="empty-state-icon">
                                        <i class="fas fa-box-open"></i>
                                    </div>
                                    <div class="empty-state-text">No packages available at the moment.</div>
                                </div>
                            </div>
                        <?php endif; ?>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation Bar -->
    <div class="bottom-navigation">
        <a href="../user/index" class="bottom-nav-item">
            <i class="fas fa-home bottom-nav-icon"></i>
            <span class="bottom-nav-text">Dashboard</span>
        </a>
        <a href="user_packages.php" class="bottom-nav-item active">
            <i class="fas fa-box bottom-nav-icon"></i>
            <span class="bottom-nav-text">Packages</span>
        </a>
        <a href="user_withdrawal.php" class="bottom-nav-item">
            <i class="fas fa-money-bill-wave bottom-nav-icon"></i>
            <span class="bottom-nav-text">Withdraw</span>
        </a>
        <a href="referrals" class="bottom-nav-item">
            <i class="fas fa-users bottom-nav-icon"></i>
            <span class="bottom-nav-text">Referrals</span>
        </a>
        <a href="settings" class="bottom-nav-item">
            <i class="fas fa-cog bottom-nav-icon"></i>
            <span class="bottom-nav-text">Settings</span>
        </a>
    </div>

    <!-- Success/Error Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="modal">
            <div class="modal-header">
                <div class="modal-icon <?= $successMessage ? 'success' : 'error' ?>">
                    <i class="fas <?= $successMessage ? 'fa-check' : 'fa-exclamation-triangle' ?>"></i>
                </div>
                <h3 class="modal-title"><?= $successMessage ? $successMessage['title'] : ($errorMessage ? 'Error' : '') ?></h3>
            </div>
            <div class="modal-body">
                <?php if ($successMessage): ?>
                    <div class="modal-details">
                        <?php foreach ($successMessage['details'] as $label => $value): ?>
                            <div class="modal-detail">
                                <span class="modal-detail-label"><?= $label ?>:</span>
                                <span class="modal-detail-value"><?= $value ?></span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <p><?= $errorMessage ?></p>
                <?php endif; ?>
            </div>
            <div class="modal-footer">
                <button class="modal-btn modal-btn-primary" id="closeModal">OK</button>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Show success/error modal if there's a message
            const messageModal = document.getElementById('messageModal');
            const closeModalBtn = document.getElementById('closeModal');
            
            <?php if ($successMessage || $errorMessage): ?>
                messageModal.classList.add('active');
            <?php endif; ?>
            
            // Close modal when OK button is clicked
            if (closeModalBtn) {
                closeModalBtn.addEventListener('click', function() {
                    messageModal.classList.remove('active');
                });
            }
            
            // Account dropdown functionality
            const accountBtn = document.getElementById('accountBtn');
            const accountDropdown = document.getElementById('accountDropdown');
            
            if (accountBtn && accountDropdown) {
                accountBtn.addEventListener('click', function(e) {
                    e.stopPropagation();
                    accountDropdown.classList.toggle('active');
                });
                
                // Close dropdown when clicking outside
                document.addEventListener('click', function() {
                    accountDropdown.classList.remove('active');
                });
                
                // Prevent dropdown from closing when clicking inside it
                accountDropdown.addEventListener('click', function(e) {
                    e.stopPropagation();
                });
            }
            
            // Right sidebar functionality
            const hamburgerBtn = document.getElementById('hamburgerBtn');
            const rightSidebar = document.getElementById('rightSidebar');
            const sidebarClose = document.getElementById('sidebarClose');
            const sidebarOverlay = document.getElementById('sidebarOverlay');
            
            if (hamburgerBtn && rightSidebar && sidebarClose && sidebarOverlay) {
                // Open sidebar when hamburger is clicked
                hamburgerBtn.addEventListener('click', function() {
                    rightSidebar.classList.add('active');
                    sidebarOverlay.classList.add('active');
                    document.body.style.overflow = 'hidden'; // Prevent body scroll when sidebar is open
                    
                    // Reset animations when opening
                    const menuItems = rightSidebar.querySelectorAll('.nk-menu-item');
                    menuItems.forEach(item => {
                        item.style.animation = 'none';
                        item.offsetHeight; // Trigger reflow
                        item.style.animation = null;
                    });
                });
                
                // Close sidebar when close button is clicked
                sidebarClose.addEventListener('click', function() {
                    rightSidebar.classList.remove('active');
                    sidebarOverlay.classList.remove('active');
                    document.body.style.overflow = ''; // Restore body scroll
                });
                
                // Close sidebar when overlay is clicked
                sidebarOverlay.addEventListener('click', function() {
                    rightSidebar.classList.remove('active');
                    sidebarOverlay.classList.remove('active');
                    document.body.style.overflow = ''; // Restore body scroll
                });
                
                // Handle submenu toggles
                const menuToggles = document.querySelectorAll('.nk-menu-toggle');
                menuToggles.forEach(toggle => {
                    toggle.addEventListener('click', function(e) {
                        e.preventDefault();
                        const parentItem = this.parentElement;
                        const submenu = parentItem.querySelector('.nk-menu-sub');
                        
                        if (submenu) {
                            submenu.classList.toggle('active');
                            this.classList.toggle('active');
                        }
                    });
                });
            }
            
            // Package tabs functionality
            const packageTabs = document.querySelectorAll('.package-tab');
            const packageContents = document.querySelectorAll('.package-content');
            
            packageTabs.forEach(tab => {
                tab.addEventListener('click', function() {
                    const tabId = this.getAttribute('data-tab');
                    
                    // Remove active class from all tabs and contents
                    packageTabs.forEach(t => t.classList.remove('active'));
                    packageContents.forEach(c => c.classList.remove('active'));
                    
                    // Add active class to clicked tab and corresponding content
                    this.classList.add('active');
                    document.getElementById(tabId + '-packages').classList.add('active');
                });
            });
            
            // Function to update all countdowns
            function updateCountdowns() {
                // Get all countdown elements
                var countdownElements = document.querySelectorAll('.countdown');
                
                countdownElements.forEach(function(el) {
                    var span = el.querySelector('span');
                    // Get the target timestamp from data attribute
                    var targetTime = parseInt(el.getAttribute('data-target'));
                    
                    // Calculate difference between now and target time
                    var now = new Date().getTime();
                    var diff = targetTime - now;
                    
                    if (diff <= 0) {
                        span.textContent = 'Processing...';
                    } else {
                        var hours = Math.floor(diff / (1000 * 60 * 60));
                        var minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                        var seconds = Math.floor((diff % (1000 * 60)) / 1000);
                        
                        // Format with leading zeros
                        hours = hours < 10 ? '0' + hours : hours;
                        minutes = minutes < 10 ? '0' + minutes : minutes;
                        seconds = seconds < 10 ? '0' + seconds : seconds;
                        
                        span.textContent = hours + ':' + minutes + ':' + seconds;
                    }
                });
            }
            
            // Update countdowns immediately
            updateCountdowns();
            
            // Update countdowns every second
            setInterval(updateCountdowns, 1000);
        });
    </script>
</body>
</html>